"""Service for composing and sending reminder emails via Gmail (robust, repo-agnostic)."""

from __future__ import annotations

from datetime import date, timedelta, datetime
from typing import Any, Dict, List, Tuple

# Local repos/services
from repositories.employee_repo import EmployeeRepository
from repositories.task_repo import TaskRepository
from repositories.assignment_repo import AssignmentRepository
from repositories.email_log_repo import EmailLogRepository
from services.settings_service import SettingsService

# Email utils
from utils.email_utils import send_email


def _get_all(repo: Any) -> List[Dict[str, Any]]:
    """Safe fetch across different repository method names."""
    for name in ("list_all", "get_all", "all"):
        if hasattr(repo, name):
            rows = getattr(repo, name)()
            return rows if isinstance(rows, list) else list(rows)
    if hasattr(repo, "table"):  # TinyDB fallback
        return [dict(r) for r in repo.table.all()]
    return []


def _to_iso(d: Any) -> str:
    if hasattr(d, "isoformat"):
        return d.isoformat()  # date/datetime
    return str(d)


def _valid_email(s: Any) -> bool:
    return isinstance(s, str) and "@" in s and "." in s


def _html_table(tasks: List[Dict[str, Any]]) -> str:
    """Minimal HTML table for the mail body."""
    rows = []
    for t in tasks:
        rows.append(
            f"<tr>"
            f"<td>{t.get('task_code','')}</td>"
            f"<td>{t.get('title','')}</td>"
            f"<td>{t.get('client_name','')}</td>"
            f"<td>{t.get('due_date','')}</td>"
            f"<td>{t.get('priority','')}</td>"
            f"</tr>"
        )
    body = (
        "<table border='1' cellspacing='0' cellpadding='6' style='border-collapse:collapse'>"
        "<thead><tr>"
        "<th>Task Code</th><th>Title</th><th>Client</th><th>Due Date</th><th>Priority</th>"
        "</tr></thead>"
        "<tbody>"
        + "".join(rows)
        + "</tbody></table>"
    )
    return body


class EmailService:
    """Builds reminder emails by joining Tasks (due in window) with Assignments & Employees."""

    def __init__(
        self,
        employee_repo: EmployeeRepository | None = None,
        task_repo: TaskRepository | None = None,
        assignment_repo: AssignmentRepository | None = None,
        email_log_repo: EmailLogRepository | None = None,
        settings_service: SettingsService | None = None,
    ) -> None:
        self.employee_repo = employee_repo or EmployeeRepository()
        self.task_repo = task_repo or TaskRepository()
        self.assignment_repo = assignment_repo or AssignmentRepository()
        self.email_log_repo = email_log_repo or EmailLogRepository()
        self.settings_service = settings_service or SettingsService()

    # ---------------- core ----------------
    def _tasks_due_window(self, start: date, end: date) -> Dict[int, Dict[str, Any]]:
        """Return {task_id: task_dict} for tasks whose due_date is in [start, end]."""
        tasks: List[Dict[str, Any]] = []
        if hasattr(self.task_repo, "tasks_due_within"):
            try:
                tasks = self.task_repo.tasks_due_within(start, end)  # type: ignore[arg-type]
            except Exception:
                tasks = _get_all(self.task_repo)
        else:
            tasks = _get_all(self.task_repo)

        out: Dict[int, Dict[str, Any]] = {}
        for t in tasks:
            due = t.get("due_date", "")
            try:
                # Accept both date strings and date objects
                if hasattr(due, "isoformat"):
                    due_str = due.isoformat()  # type: ignore[assignment]
                else:
                    due_str = str(due)
                y, m, d = map(int, due_str[:10].split("-"))
                due_dt = date(y, m, d)
                if start <= due_dt <= end:
                    out[int(t.get("id"))] = t
            except Exception:
                tid = t.get("id")
                if isinstance(tid, int):
                    out[tid] = t
        return out

    def _log_email(self, payload: Dict[str, Any]) -> None:
        """Insert into email log (works whether repo exposes create_log or just table)."""
        data = dict(payload)
        data.setdefault("sent_at", _to_iso(datetime.now()))
        if hasattr(self.email_log_repo, "create_log"):
            try:
                self.email_log_repo.create_log(data)  # type: ignore[attr-defined]
                return
            except Exception:
                pass
        if hasattr(self.email_log_repo, "table"):
            self.email_log_repo.table.insert(data)  # type: ignore[attr-defined]

    # ---------------- API ----------------
    def send_due_reminders(self, look_ahead_days: int | None = None) -> Dict[str, Any]:
        """
        Send reminder emails:
          - window: today .. today+look_ahead_days
          - only for active employees having a valid email
          - only assignments whose tasks are due within the window
          - ignore assignments with status in ('Completed', 'Deferred')
        Returns a summary dict.
        """
        raw_settings = self.settings_service.get_settings()
        # Handle dict or Pydantic model
        if look_ahead_days is None:
            if isinstance(raw_settings, dict):
                lookahead = int(raw_settings.get("look_ahead_days", 7))
            else:
                lookahead = int(getattr(raw_settings, "look_ahead_days", 7))
        else:
            lookahead = int(look_ahead_days)

        today = date.today()
        end = today + timedelta(days=lookahead)

        # Load data
        employees = _get_all(self.employee_repo)
        assignments = _get_all(self.assignment_repo)
        tasks_in_window = self._tasks_due_window(today, end)

        # Index helpers
        emp_by_id: Dict[int, Dict[str, Any]] = {int(e["id"]): e for e in employees if "id" in e}
        # Filter assignments -> due window & pending
        effective: Dict[int, List[Dict[str, Any]]] = {}
        for a in assignments:
            try:
                if a.get("status") in ("Completed", "Deferred"):
                    continue
                tid = int(a.get("task_id"))
                eid = int(a.get("employee_id"))
            except Exception:
                continue
            if tid not in tasks_in_window:
                continue
            effective.setdefault(eid, []).append(a)

        # Send
        sent = 0
        skipped = 0
        details: List[Tuple[int, str, int]] = []  # (employee_id, email, count)

        for eid, asgs in effective.items():
            emp = emp_by_id.get(eid)
            if not emp or not emp.get("is_active", True):
                skipped += 1
                continue
            email = emp.get("email", "")
            if not _valid_email(email):
                skipped += 1
                continue

            # Build task rows for this employee
            rows: List[Dict[str, Any]] = []
            for a in asgs:
                t = tasks_in_window.get(int(a["task_id"]))
                if not t:
                    continue
                rows.append(
                    {
                        "task_code": t.get("task_code", ""),
                        "title": t.get("title", ""),
                        "client_name": t.get("client_name", ""),
                        "due_date": t.get("due_date", ""),
                        "priority": t.get("priority", ""),
                    }
                )

            if not rows:
                skipped += 1
                continue

            # Sort by due_date ascending
            try:
                rows.sort(key=lambda r: r.get("due_date", ""))
            except Exception:
                pass

            emp_name = emp.get("full_name", "Employee")
            subject = f"Task Reminders — Next {lookahead} Days — {emp_name} — {today.strftime('%d %b %Y')}"
            body = (
                f"<p>Dear {emp_name},</p>"
                "<p>Please find below the tasks due in the upcoming period:</p>"
                f"{_html_table(rows)}"
                "<p>Regards,<br>Task Manager</p>"
            )

            try:
                ok = send_email(to=email, subject=subject, html_body=body)
                status = "SENT" if ok else "FAILED"
            except Exception as e:
                status = f"ERROR: {e}"

            # Log the attempt
            self._log_email(
                {
                    "employee_id": eid,
                    "email": email,
                    "subject": subject,
                    "body": body,
                    "status": status,
                    "sent_at": _to_iso(datetime.now()),
                }
            )

            if status == "SENT":
                sent += 1
                details.append((eid, str(email), len(rows)))
            else:
                skipped += 1

        return {
            "window_start": _to_iso(today),
            "window_end": _to_iso(end),
            "lookahead_days": lookahead,
            "emails_attempted": len(effective),
            "emails_sent": sent,
            "emails_skipped_or_failed": skipped,
            "details": details,  # list of (employee_id, email, task_count)
        }
